// SPDX-License-Identifier: CC0-1.0
//
// SPDX-FileContributor: Antonio Niño Díaz, 2024-2025

// This example sets up a DMA channel to copy data to horizontal scroll
// registers during every horizontal blanking period. It changes the horizontal
// scroll of a background to generate a wave effect.
//
// The 3D layer is displayed using video layer 0. That means the 3D output can
// also be scrolled with REG_BG0HOFS and REG_BG0VOFS. However, some emulators
// won't do a correct screen capture if the 3D output is scrolled while the
// capture takes place.
//
// NOTE: Be very careful with DMA! If you call irqSet() to set the VBL handler
// at the beginning of main(), the DMA copies will be configured by the time a
// 3D texture is loaded. The texture is loaded with DMA, and it will cause a
// hang. The texture takes more than one scaline to be copied, so the HBL DMA
// copy is triggered, and it hangs the DMA completely.

#include <stdio.h>

#include <nds.h>

// Headers autogenerated for each PNG file inside GFXDIRS in the Makefile
#include "forest_town.h"
#include "neon.h"

static int angle_offset = 0;

static uint16_t bg0ofs_table[192];
static uint16_t bg2ofs_table[192];

static void vblank_handler(void)
{
    // Stop the previous DMA copies
    dmaStopSafe(0);
    dmaStopSafe(1);

    // Fill tables
    for (int i = 0; i < 192; i++)
    {
        int32_t value = sinLerp(degreesToAngle(angle_offset + i + 1)) >> 7;
        bg0ofs_table[i] = value;
        bg2ofs_table[i] = -value;
    }

    // Set the horizontal scroll for the first line. The first horizontal blank
    // happens after line 0 has been drawn, so we need to set the scroll of line
    // 0 now.
    REG_BG0HOFS = bg0ofs_table[0];
    REG_BG2HOFS = bg2ofs_table[0];

    // Make sure that DMA can see the updated values of the arrays and the
    // updated values don't stay in the data cache.
    DC_FlushRange(bg0ofs_table, sizeof(bg0ofs_table));
    DC_FlushRange(bg2ofs_table, sizeof(bg2ofs_table));

    // Restart the DMA copies
    dmaSetParams(0,
                 &bg0ofs_table[1], // Skip first entry (we have just used it)
                 (void *)&REG_BG0HOFS, // Write to horizontal scroll register
                 DMA_SRC_INC | // Autoincrement source after each copy
                 DMA_DST_FIX | // Keep destination fixed
                 DMA_START_HBL | // Start copy at the start of horizontal blank
                 DMA_REPEAT | // Don't stop DMA after the first copy.
                 DMA_COPY_HALFWORDS | 1 | // Copy one halfword each time
                 DMA_ENABLE);

    dmaSetParams(1, &bg2ofs_table[1], (void *)&REG_BG2HOFS,
                 DMA_SRC_INC | DMA_DST_FIX | DMA_START_HBL | DMA_REPEAT |
                 DMA_COPY_HALFWORDS | 1 | DMA_ENABLE);

    // Note that writing to the scroll registers directly will overwrite the
    // values writen by the user with bgSetScroll() and bgUpdate().
}

int main(int argc, char **argv)
{
    // Enable 3D and sprites
    videoSetMode(MODE_0_3D);

    // Setup some VRAM as memory for main engine background and 3D textures.
    vramSetPrimaryBanks(VRAM_A_MAIN_BG, VRAM_B_LCD, VRAM_C_TEXTURE, VRAM_D_TEXTURE);

    // Setup console in the sub screen
    vramSetBankH(VRAM_H_SUB_BG);
    consoleDemoInit();

    // Setup 2D background
    // ===================

    int bg = bgInit(2, BgType_Text8bpp, BgSize_T_256x256, 0, 4);
    bgSetPriority(bg, 2);

    dmaCopy(forest_townTiles, bgGetGfxPtr(bg), forest_townTilesLen);
    dmaCopy(forest_townMap, bgGetMapPtr(bg), forest_townMapLen);
    dmaCopy(forest_townPal, BG_PALETTE, forest_townPalLen);

    // Setup 3D
    // ========

    glInit();

    glEnable(GL_TEXTURE_2D);
    glEnable(GL_ANTIALIAS);

    // IMPORTANT: The 3D background must be transparent (alpha = 0) so that the
    // 2D layers behind it can be seen.
    glClearColor(0, 0, 0, 0);
    glClearPolyID(63);

    glClearDepth(0x7FFF);

    glViewport(0, 0, 255, 191);

    int textureID;

    // Load texture
    glGenTextures(1, &textureID);
    glBindTexture(0, textureID);
    if (glTexImage2D(0, 0, GL_RGBA, 128, 128, 0, TEXGEN_TEXCOORD, neonBitmap) == 0)
    {
        printf("Failed to load texture\n");
        while (1)
            swiWaitForVBlank();
    }

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluPerspective(70, 256.0 / 192.0, 0.1, 40);

    gluLookAt(0.0, 0.0, 2.0,  // Position
              0.0, 0.0, 0.0,  // Look at
              0.0, 1.0, 0.0); // Up

    // Start special efffect
    // =====================

    irqSet(IRQ_VBLANK, vblank_handler);

    // Setup done
    // ==========

    int angle_x = 0;
    int angle_z = 0;

    while (1)
    {
        // Synchronize game loop to the screen refresh
        swiWaitForVBlank();

        // Print some text in the demo console
        // -----------------------------------

        consoleClear();

        // Print some controls
        printf("PAD:   Rotate quad\n");
        printf("START: Exit to loader\n");
        printf("\n");

        // Handle user input
        // -----------------

        scanKeys();

        uint16_t keys = keysHeld();

        if (keys & KEY_LEFT)
            angle_z += 3;
        if (keys & KEY_RIGHT)
            angle_z -= 3;

        if (keys & KEY_UP)
            angle_x += 3;
        if (keys & KEY_DOWN)
            angle_x -= 3;

        if (keys & KEY_START)
            break;

        // Animate wave effect
        // -------------------

        angle_offset++;
        if (angle_offset >= degreesToAngle(360))
            angle_offset -= degreesToAngle(360);

        // Render 3D scene
        // ---------------

        glMatrixMode(GL_MODELVIEW);

        glPushMatrix();

        glRotateZ(angle_z);
        glRotateX(angle_x);

        glPolyFmt(POLY_ALPHA(31) | POLY_CULL_NONE);

        glBindTexture(0, textureID);

        glColor3f(1, 1, 1);

        glBegin(GL_QUADS);

            glTexCoord2t16(0, inttot16(128));
            glVertex3v16(floattov16(-1), floattov16(-1), 0);

            glTexCoord2t16(inttot16(128), inttot16(128));
            glVertex3v16(floattov16(1), floattov16(-1), 0);

            glTexCoord2t16(inttot16(128), 0);
            glVertex3v16(floattov16(1), floattov16(1), 0);

            glTexCoord2t16(0, 0);
            glVertex3v16(floattov16(-1), floattov16(1), 0);

        glEnd();

        glPopMatrix(1);

        glFlush(0);
    }

    glDeleteTextures(1, &textureID);

    return 0;
}

